/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

        Key filtering routines implementing the PGPFilterRef abstract data type

        $Id: pgpKeyFilter.c,v 1.28 2001/01/25 22:11:09 jeffc Exp $
____________________________________________________________________________*/
#include "pgpConfig.h"

#include <string.h>
#include <ctype.h>

#include "pgpDebug.h"
#include "pgpErrors.h"
#include "pgpContext.h"
#include "pgpKeyPriv.h"
#include "pgpKeyFilterPriv.h"
#include "pgpPubKey.h"
#include "pgpTimeDate.h"
#include "pgpX509Priv.h"

#define PGPValidateMatchCriterion( m ) \
	pgpAssert( (m) == kPGPMatchCriterion_Equal || \
			   (m) == kPGPMatchCriterion_GreaterOrEqual || \
			   (m) == kPGPMatchCriterion_LessOrEqual || \
			   (m) == kPGPMatchCriterion_SubString )
	

#define sCompareBoolean(f,v) (!(f)->value.propbool.val == !(v))
#define sCompareNumber(f,v) sComparisonMatchesCriterion( \
								(v) - (f)->value.propnum.val, (f)->match )


	PGPBoolean
PGPFilterIsValid( PGPFilterRef	filter )
{
	return( IsntNull( filter ) && filter->magic == kPGPFilterMagic );
}


	
/*
 * Stolen from pgpRngPub.c:
 *
 * Return pointer to first instance of (s1,l1) in (s0,l0),
 * ignoring case.  Uses a fairly simple-minded algorithm.
 * Search for the first char of s1 in s0, and when we have it,
 * scan for the rest.
 *
 * Is it worth mucking with Boyer-Moore or the like?
 */
static char const *
xmemimem(char const *s0, size_t l0, char const *s1, size_t l1)
{
	char c0, c1, c2;
	size_t l;

	/*
	 * The trivial cases - this means that NULL inputs are very legal
	 * if the correspinding lengths are zero.
	 */
	if (l0 < l1)
		return NULL;
	if (!l1)
		return s0;
	l0 -= l1;

	c1 = tolower((unsigned char)*s1);
	do {
		c0 = tolower((unsigned char)*s0);
		if (c0 == c1) {
			l = 0;
			do {
				if (++l == l1)
					return s0;
				c0 = tolower((unsigned char)s0[l]);
				c2 = tolower((unsigned char)s1[l]);
			} while (c0 == c2);
		}
		s0++;
	} while (l0--);
	return NULL;
}


	static PGPError
sAllocateFilter(
	PGPContextRef		context, 
	PGPFilterType		filterType,
	PGPMatchCriterion	match,
	PGPFilterRef *		outFilter)
{
	PGPFilterRef	newFilter	= NULL;
	PGPError		err	= kPGPError_NoErr;

	PGPValidatePtr( outFilter );
	*outFilter	= NULL;
	PGPValidateContext( context );
	PGPValidateParam( filterType < kPGPFilterTypeEnd );
	PGPValidateMatchCriterion( match );
	
	newFilter	= (PGPFilterRef)
		pgpContextMemAlloc( context, sizeof( PGPFilter),
		kPGPMemoryMgrFlags_Clear);
	if( IsntNull( newFilter ) )
	{
		newFilter->magic		= kPGPFilterMagic;
		newFilter->context		= context;
		newFilter->refCount		= 1;
		newFilter->filterType	= filterType;
		newFilter->match		= match;
	}
	else
	{
		err	= kPGPError_OutOfMemory;
	}

	
	*outFilter = newFilter;
	pgpAssertErrWithPtr( err, *outFilter );
	return err;
}


	PGPError
PGPIncFilterRefCount( PGPFilterRef filter )
{
	PGPValidateFilter( filter );

	pgpEnterPGPErrorFunction();

	filter->refCount++;
	return kPGPError_NoErr;
}

	PGPError
PGPFilterChildObjects( PGPFilterRef filter, PGPBoolean filterChildren )
{
	PGPValidateFilter( filter );

	pgpEnterPGPErrorFunction();

	filter->filterChildren = filterChildren;
	return kPGPError_NoErr;
}


	PGPError
PGPFreeFilter( PGPFilterRef filter )
{
	PGPError	err	= kPGPError_NoErr;
	
	PGPValidateFilter( filter );
	
	pgpEnterPGPErrorFunction();

	pgpAssert( filter->refCount >= 1 );
	
	filter->refCount--;
	if( filter->refCount == 0 ) 
	{
		PGPContextRef	context		= filter->context;
		void *			ptrToFree	= NULL;
		
		switch( filter->filterType) 
		{
		default:
			break;
		case kPGPFilterTypeKeyKeyID:
			break;
		case kPGPFilterTypeKeySubKeyID:
			break;
		case kPGPFilterTypeKeyFingerPrint:
			ptrToFree	= filter->value.keyFingerPrint.keyFingerPrintData;
			break;
		case kPGPFilterTypeUserIDEmail:
			ptrToFree	= filter->value.userIDEmail;
			break;
		case kPGPFilterTypeUserIDName:
			ptrToFree	= filter->value.userIDName;
			break;
		case kPGPFilterTypeUserIDString:
			ptrToFree	= filter->value.userIDString;
			break;
		case kPGPFilterTypeSigKeyID:
			break;
		case kPGPFilterTypeKeyBuffer:
		case kPGPFilterTypeSubKeyBuffer:
		case kPGPFilterTypeUserIDBuffer:
		case kPGPFilterTypeSigBuffer:
			ptrToFree	= filter->value.propbuffer.val;
			break;
		case kPGPFilterTypeNot:
			PGPFreeFilter( filter->value.notFilter );
			break;
		case kPGPFilterTypeAnd:
			PGPFreeFilter( filter->value.andFilter.andFilter1 );
			PGPFreeFilter( filter->value.andFilter.andFilter2 );
			break;
		case kPGPFilterTypeOr:
			PGPFreeFilter( filter->value.orFilter.orFilter1 );
			PGPFreeFilter( filter->value.orFilter.orFilter2 );
			break;
		}
		
		if ( IsntNull( ptrToFree ) )
		{
			pgpContextMemFree( context, ptrToFree);
		}
		
		pgpContextMemFree( context, filter );
	}
	
	return err;
}


/*____________________________________________________________________________
	The passed in filter will be freed when the newly created filter is
	freed.  Therefore, if an error occurs, the passed filter must
	also be freed.  Caller should increment the passed-in filter 
	ref count if the filter should persist.
____________________________________________________________________________*/
	PGPError
PGPNegateFilter(
	PGPFilterRef	filter,
	PGPFilterRef *	outFilter)
{
	PGPError		err	= kPGPError_NoErr;

	if ( IsntNull( outFilter ) )
	{
		*outFilter	= NULL;
		if ( ! PGPFilterIsValid( filter ) )
			err	= kPGPError_BadParams;
	}
	else
	{
		err	= kPGPError_BadParams;
	}
		
	pgpEnterPGPErrorFunction();

	if ( IsntPGPError( err ) )
	{
		err = sAllocateFilter( filter->context, kPGPFilterTypeNot,
							   kPGPMatchCriterion_Equal, outFilter);
		pgpAssertErrWithPtr( err, *outFilter );
								
		if ( IsntPGPError( err ) )
		{
			(*outFilter)->value.notFilter	= filter;
		}
	}
	
	/* careful to clean up in event of error */
	if ( IsPGPError( err ) )
	{
		if ( PGPFilterIsValid( filter ) )
		{
			PGPFreeFilter( filter );
			filter	= NULL;
		}
	}
	
	pgpAssertErrWithPtr( err, *outFilter );
	return err;
}


/*____________________________________________________________________________
	The passed in filters will be freed when the newly created filter is
	freed.  Therefore, if an error occurs, the passed filters must
	also be freed.  Caller should increment the passed-in filter 
	ref count if the filters should persist.
____________________________________________________________________________*/
	PGPError
PGPIntersectFilters(
	PGPFilterRef 	filter1,
	PGPFilterRef	filter2, 
	PGPFilterRef *	outFilter)
{
	PGPError			err	= kPGPError_NoErr;
		
	if ( IsntNull( outFilter ) )
		*outFilter	= NULL;
			
	pgpEnterPGPErrorFunction();

	if ( IsntNull( outFilter ) &&
		PGPFilterIsValid( filter1 ) &&
		PGPFilterIsValid( filter2 )  )
	{
		err = sAllocateFilter( filter1->context, kPGPFilterTypeAnd,
							   kPGPMatchCriterion_Equal, outFilter);
		if( IsntPGPError( err ) )
		{
			(*outFilter)->value.andFilter.andFilter1 = filter1;
			(*outFilter)->value.andFilter.andFilter2 = filter2;
		}
	}
	else
	{
		err	= kPGPError_BadParams;
	}
	
	
	/* careful to clean up in event of error */
	if ( IsPGPError( err ) )
	{
		if ( PGPFilterIsValid( filter1 ) )
			PGPFreeFilter( filter1 );
		if ( PGPFilterIsValid( filter2 ) )
			PGPFreeFilter( filter2 );
	}
	
	pgpAssertErrWithPtr( err, *outFilter );
	return err;
}


/*____________________________________________________________________________
	The passed in filters will be freed when the newly created filter is
	freed.  Therefore, if an error occurs, the passed filters must
	also be freed.  Caller should increment the passed-in filter 
	ref count if the filters should persist.
____________________________________________________________________________*/
	PGPError
PGPUnionFilters(
	PGPFilterRef	filter1,
	PGPFilterRef	filter2, 
	PGPFilterRef *	outFilter)
{
	PGPError		err	= kPGPError_NoErr;

	if ( IsntNull( outFilter ) )
		*outFilter	= NULL;
		
	pgpEnterPGPErrorFunction();

	if ( IsntNull( outFilter ) &&
		PGPFilterIsValid( filter1 ) &&
		PGPFilterIsValid( filter2 )  )
	{
		err = sAllocateFilter( filter1->context, kPGPFilterTypeOr,
							   kPGPMatchCriterion_Equal,  outFilter);
		if( IsntPGPError( err ) )
		{
			(*outFilter)->value.orFilter.orFilter1 = filter1;
			(*outFilter)->value.orFilter.orFilter2 = filter2;
		}
	}
	else
	{
		err	= kPGPError_BadParams;
	}
	
	/* careful to clean up in event of error */
	if ( IsPGPError( err ) )
	{
		if ( PGPFilterIsValid( filter1 ) )
			PGPFreeFilter( filter1 );
		if ( PGPFilterIsValid( filter2 ) )
			PGPFreeFilter( filter2 );
	}
	
	pgpAssertErrWithPtr( err, *outFilter );
	return err;
}


/********************** Generic property filters ************************/

#define kPGPKeyProperty_FirstProperty		100
#define kPGPSubKeyProperty_FirstProperty	500
#define kPGPUserIDProperty_FirstProperty	900
#define kPGPSigProperty_FirstProperty		1300


	PGPError
PGPNewKeyDBObjBooleanFilter(
	PGPContextRef		context,
	PGPKeyDBObjProperty		prop,
	PGPBoolean			match,
	PGPFilterRef *		outFilter)
{
	PGPError		err	= kPGPError_NoErr;
	PGPFilterType	proptype;

	PGPValidatePtr( outFilter );
	*outFilter	= NULL;
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	if( prop < kPGPSubKeyProperty_FirstProperty )
		proptype = kPGPFilterTypeKeyBoolean;
	else if( prop < kPGPUserIDProperty_FirstProperty )
		proptype = kPGPFilterTypeSubKeyBoolean;
	else if( prop < kPGPSigProperty_FirstProperty )
		proptype = kPGPFilterTypeUserIDBoolean;
	else
		proptype = kPGPFilterTypeSigBoolean;

	err = sAllocateFilter( context, proptype,  kPGPMatchCriterion_Equal,
					outFilter);
	if( IsntPGPError( err ) )
	{
		(*outFilter)->value.propbool.prop = prop;
		(*outFilter)->value.propbool.val = match;
	}
	
	pgpAssertErrWithPtr( err, *outFilter );
	return err;
}

	PGPError
PGPNewKeyDBObjNumericFilter(
	PGPContextRef		context,
	PGPKeyDBObjProperty		prop,
	PGPUInt32			val,
	PGPMatchCriterion	match,
	PGPFilterRef *		outFilter)
{
	PGPError		err	= kPGPError_NoErr;
	PGPFilterType	proptype;

	PGPValidatePtr( outFilter );
	*outFilter	= NULL;
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	if( prop < kPGPSubKeyProperty_FirstProperty )
		proptype = kPGPFilterTypeKeyNumber;
	else if( prop < kPGPUserIDProperty_FirstProperty )
		proptype = kPGPFilterTypeSubKeyNumber;
	else if( prop < kPGPSigProperty_FirstProperty )
		proptype = kPGPFilterTypeUserIDNumber;
	else
		proptype = kPGPFilterTypeSigNumber;

	err = sAllocateFilter( context, proptype, match, outFilter);
	if( IsntPGPError( err ) )
	{
		(*outFilter)->value.propnum.prop = prop;
		(*outFilter)->value.propnum.val = val;
	}
	
	pgpAssertErrWithPtr( err, *outFilter );
	return err;
}

	PGPError
PGPNewKeyDBObjTimeFilter(
	PGPContextRef		context,
	PGPKeyDBObjProperty		prop,
	PGPTime				val,
	PGPMatchCriterion	match,
	PGPFilterRef *		outFilter)
{
	PGPError		err	= kPGPError_NoErr;
	PGPFilterType	proptype;

	PGPValidatePtr( outFilter );
	*outFilter	= NULL;
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	if( prop < kPGPSubKeyProperty_FirstProperty )
		proptype = kPGPFilterTypeKeyTime;
	else if( prop < kPGPUserIDProperty_FirstProperty )
		proptype = kPGPFilterTypeSubKeyTime;
	else if( prop < kPGPSigProperty_FirstProperty )
		proptype = kPGPFilterTypeUserIDTime;
	else
		proptype = kPGPFilterTypeSigTime;

	err = sAllocateFilter( context, proptype, match, outFilter);
	if( IsntPGPError( err ) )
	{
		(*outFilter)->value.proptime.prop = prop;
		(*outFilter)->value.proptime.val = val;
	}
	
	pgpAssertErrWithPtr( err, *outFilter );
	return err;
}

	PGPError
PGPNewKeyDBObjDataFilter(
	PGPContextRef		context,
	PGPKeyDBObjProperty		prop,
	const void			   *val,
	PGPSize				len,
	PGPMatchCriterion	match,
	PGPFilterRef *		outFilter)
{
	PGPError		err	= kPGPError_NoErr;
	PGPFilterType	proptype;
	PGPFilterRef	newFilter;

	PGPValidatePtr( outFilter );
	*outFilter	= NULL;
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	if( prop < kPGPSubKeyProperty_FirstProperty )
		proptype = kPGPFilterTypeKeyBuffer;
	else if( prop < kPGPUserIDProperty_FirstProperty )
		proptype = kPGPFilterTypeSubKeyBuffer;
	else if( prop < kPGPSigProperty_FirstProperty )
		proptype = kPGPFilterTypeUserIDBuffer;
	else
		proptype = kPGPFilterTypeSigBuffer;


	err = sAllocateFilter( context, proptype, match, &newFilter);
	if( IsntPGPError( err ) )
	{
		PGPByte *		value;

		value = (PGPByte *)pgpContextMemAlloc( context, len, 0);
		if( IsntNull( value ) )
		{
			pgpCopyMemory (val, value, len);
			newFilter->value.propbuffer.prop = prop;
			newFilter->value.propbuffer.val = value;
			newFilter->value.propbuffer.len = len;
		}
		else
		{
			PGPFreeFilter( newFilter );
			newFilter	= NULL;
			err	= kPGPError_OutOfMemory;
		}
	}
	
	*outFilter = newFilter;
	pgpAssertErrWithPtr( err, *outFilter );
	return err;
}



#if NOT_YET		/* [ */

/*
 * Create a filtered KeySet based on the same KeyDB as the specified
 * original.  The original must be unfiltered.  The filtering type
 * is chosen automatically based on the string.
 *
 * Currently, a "0x" prefix looks up by keyID, otherwise
 * by userid name substring matching.
 */
	PGPError 
PGPNewKeyFilterFromStringQuery(
	PGPContextRef		context,
	char const *		query,
	PGPFilterRef *		outFilter)
{
	PGPError		err	= kPGPError_NoErr;
	PGPFilterRef	filter = kInvalidPGPKeySetRef;

	PGPValidatePtr( outFilter );
	PGPValidatePtr( query );
	*outFilter	= NULL;
	PGPValidateContext( context );
	
not yet

	if (query[0] == '0' && query[1] == 'x')
	{
		PGPKeyIDRef		keyID;

		err = PGPNewKeyIDFromString( context, query, &keyID );
		if ( IsntPGPError( err ) )
		{
			err = PGPNewKeyIDFilter( context, keyID, &filter );
			PGPFreeKeyID( keyID );
		}
	}
	else
	{
		err = PGPNewUserIDStringFilter( context, query,
										kPGPMatchCriterion_SubString,
										&filter );
	}

	*outFilter	= filter;
	pgpAssertErrWithPtr( err, *outFilter );
	return err;
}

#endif	/* ] NOT_YET */

	static PGPBoolean
sComparisonMatchesCriterion(
	long				comparison,
	PGPMatchCriterion	criterion )
{
	switch (criterion)
	{
		case kPGPMatchCriterion_LessOrEqual:
			return comparison <= 0;
		case kPGPMatchCriterion_GreaterOrEqual:
			return comparison >= 0;
		case kPGPMatchCriterion_Equal:
			return comparison == 0;
		default:
			pgpAssert(0);
			break;
	}
	return FALSE;
}


	static PGPBoolean
sFilterUserID(
	PGPFilterRef		filter,
	PGPKeyDBObj		   *userid,
	char const *		string,
	PGPSize				stringLength)
{
	PGPBoolean			result = FALSE;
	char const *		nameStr;
	PGPSize				nameLength;
	char const *		p;

	nameStr = pgpUserIDName(userid, &nameLength);
	switch (filter->filterType)
	{
		case kPGPFilterTypeUserIDEmail:
			p = (char *)memchr(nameStr, '<', nameLength);
			if (p == NULL)
			{
				nameLength = 0;
				break;
			}
			p++;
			nameLength -= (p - nameStr);
			nameStr = p;
			p = (char *)memchr(nameStr, '>', nameLength);
			if (p != NULL)
				nameLength = p - nameStr;
			break;
		case kPGPFilterTypeUserIDName:
			p = (char *)memchr(nameStr, '<', nameLength);
			if (p == NULL)
				break;
			while (p > nameStr && p[-1] == ' ')
				p--;
			nameLength = p - nameStr;
			break;
		case kPGPFilterTypeUserIDString:
			break;
		default:
			/* This should never happen */
			pgpAssert(0);
			break;
	}
	if (filter->match == kPGPMatchCriterion_Equal)
	{
		if (nameLength == stringLength &&
			xmemimem(nameStr, nameLength,
				string, stringLength) != NULL)
		{
			result = TRUE;
		}
	}
	else if (filter->match == kPGPMatchCriterion_SubString)
	{
		if (xmemimem(nameStr, nameLength,
					 string, stringLength))
		{
			result = TRUE;
		}
	}
	return result;
}


/* Find the object or a parent of the specified type, or return NULL */
	static PGPKeyDBObj *
sFindParent( PGPKeyDBObj *obj, PGPUInt32 type )
{
	PGPUInt32		objtype;

	pgpAssert( type != RINGTYPE_KEY );
	while( (objtype = pgpObjectType( obj )) != type )
	{
		if( objtype == RINGTYPE_KEY )
			return NULL;
		obj = obj->up;
	}
	return obj;
}



/* Compare the time from the key object to the time from the filter */
	static PGPBoolean
sCompareTime(
			PGPFilterRef		filter,
			PGPTime				keyTime)
{
	PGPBoolean	result = FALSE;
	PGPTime		filterTime = filter->value.proptime.val;

	if (keyTime == filterTime)
		result = TRUE;	/* All match criteria include equality */
	else if (filter->match == kPGPMatchCriterion_Equal)
		result = FALSE;
	else
	{
		/*
		 * Now we know that the two times are unequal,
		 * and the match criterion is either <= or >=.
		 * So we'll just evaluate <=, and then invert
		 * the result if the criterion was >=.
		 */
		
		result = (keyTime <= filterTime);

		if( filter->filterType == kPGPFilterTypeKeyTime &&
		   (filter->value.proptime.prop == kPGPKeyProperty_Expiration ||
			filter->value.proptime.prop == kPGPSubKeyProperty_Expiration ) )
		{
			if (filterTime == kPGPExpirationTime_Never)
				result = TRUE;
			else if (keyTime == kPGPExpirationTime_Never)
				result = FALSE;
		}
		
		if (filter->match == kPGPMatchCriterion_GreaterOrEqual)
		{
			result = !result;
		}
		else
		{
			pgpAssert(filter->match == kPGPMatchCriterion_LessOrEqual);
		}
	}
	return result;
}

/* Comparison structure for string buffers - must be equal or substring */
	static PGPBoolean
sCompareString(
			PGPFilterRef		filter,
			void			   *bufval,
			PGPSize				buflen)
{
	PGPBoolean result;

	if( filter->value.propbuffer.prop == kPGPKeyProperty_KeyID ||
		filter->value.propbuffer.prop == kPGPSubKeyProperty_KeyID ||
		filter->value.propbuffer.prop == kPGPSigProperty_KeyID )
	{
		pgpAssert( filter->match == kPGPMatchCriterion_Equal );
		
		result = pgpKeyIDsEqual( filter->value.propbuffer.val, bufval );
	}
	else
	{
		if (filter->match == kPGPMatchCriterion_Equal)
		{
			result = (buflen == filter->value.propbuffer.len) &&
					 pgpMemoryEqual( bufval, filter->value.propbuffer.val,
									 buflen );
		}
		else
		{
			pgpAssert (filter->match == kPGPMatchCriterion_SubString);
			result = (NULL != xmemimem( bufval, buflen,
										filter->value.propbuffer.val,
										filter->value.propbuffer.len ) );
		}
	}
	return result;
}



	PGPBoolean
pgpKeyDBObjMatchesFilter(
	PGPFilterRef		filter,
	PGPKeyDBObjRef		obj)
{
	PGPBoolean			boolval;
	PGPUInt32			numval;
	PGPTime				timeval;
	PGPByte			   *bufval;
	PGPSize				buflen;
	PGPBoolean			result = FALSE;
	PGPKeyDBObj		   *userid;
	PGPKeyDBObj		   *sig;
	PGPKeyDBObj		   *subkey;
	PGPKeyDBObj		   *key;
	PGPByte				pkAlg;
	PGPKeyID			keyID;
	PGPInt32			comparison;
	PGPError			err = kPGPError_NoErr;

	if ( ! PGPFilterIsValid( filter ) )
		return( FALSE );
	
	/* key object will be top level key (possibly same as obj) */
	for( key = obj; pgpObjectType(key) != RINGTYPE_KEY; key = key->up )
		;

	/* Shortcut to detect child validity if we've checked parent key */
	if( pgpObjectType( obj ) != RINGTYPE_KEY
		&& IsntNull( filter->cachedKeyQuery ) )
	{
		if( key == filter->cachedKeyQuery )
		{
			if( !filter->filterChildren )
				return filter->cachedResult;
			if( !filter->cachedResult )
				return FALSE;
			/* Continue on if we're filtering children and parent key was OK */
		}
	}

	switch(filter->filterType) 
	{
		/* Key property filters */
		case kPGPFilterTypeKeyBoolean:
			err = pgpGetKeyBoolean (key,
							(PGPKeyDBObjProperty)filter->value.propbool.prop,
							&boolval);
			if( IsntPGPError( err ) )
				result = sCompareBoolean( filter, boolval );
			break;
		case kPGPFilterTypeKeyNumber:
			err = pgpGetKeyNumber (key,
							(PGPKeyDBObjProperty)filter->value.propnum.prop,
							(PGPInt32 *)&numval);
			if( IsntPGPError( err ) )
				result = sCompareNumber( filter, numval );
			break;
		case kPGPFilterTypeKeyTime:
			err = pgpGetKeyTime (key,
							(PGPKeyDBObjProperty)filter->value.proptime.prop,
							&timeval);
			if( IsntPGPError( err ) )
				result = sCompareTime( filter, timeval );
			break;
		case kPGPFilterTypeKeyBuffer:
			err = pgpGetKeyAllocatedPropertyBuffer( key,
							(PGPKeyDBObjProperty)filter->value.propbuffer.prop,
							&bufval, &buflen );
			if( IsPGPError( err ) )
				break;
			result = sCompareString( filter, bufval, buflen );
			PGPFreeData( bufval );
			break;
		case kPGPFilterTypeSubKeyBoolean:
			if( filter->filterChildren
				&& IsntNull( subkey = sFindParent( obj, RINGTYPE_SUBKEY ) ) )
			{
				err = pgpGetSubKeyBoolean (subkey,
							(PGPKeyDBObjProperty)filter->value.propbool.prop,
							&boolval);
				if( IsntPGPError( err ) )
					result = sCompareBoolean( filter, boolval );
			} else for (subkey = key->down; !result && IsntNull( subkey );
				 				subkey = subkey->next)
			{
				if (!pgpKeyDBObjIsReal( subkey ) )
					continue;
				if( pgpObjectType( subkey ) != RINGTYPE_SUBKEY )
					continue;
				err = pgpGetSubKeyBoolean (subkey,
							(PGPKeyDBObjProperty)filter->value.propbool.prop,
							&boolval);
				if( IsntPGPError( err ) )
					result = sCompareBoolean( filter, boolval );
			}
			break;
		case kPGPFilterTypeSubKeyNumber:
			if( filter->filterChildren
				&& IsntNull( subkey = sFindParent( obj, RINGTYPE_SUBKEY ) ) )
			{
				err = pgpGetSubKeyNumber (subkey,
							(PGPKeyDBObjProperty)filter->value.propnum.prop,
							(PGPInt32 *)&numval);
				if( IsntPGPError( err ) )
					result = sCompareNumber( filter, numval );
			} else for (subkey = key->down; !result && IsntNull(subkey);
							 	subkey = subkey->next)
			{
				if (!pgpKeyDBObjIsReal( subkey ) )
					continue;
				if( pgpObjectType( subkey ) != RINGTYPE_SUBKEY )
					continue;
				err = pgpGetSubKeyNumber (subkey,
							(PGPKeyDBObjProperty)filter->value.propnum.prop,
							(PGPInt32 *)&numval);
				if( IsntPGPError( err ) )
					result = sCompareNumber( filter, numval );
			}
			break;
		case kPGPFilterTypeSubKeyTime:
			if( filter->filterChildren
				&& IsntNull( subkey = sFindParent( obj, RINGTYPE_SUBKEY ) ) )
			{
				err = pgpGetSubKeyTime (subkey,
							(PGPKeyDBObjProperty)filter->value.proptime.prop,
							&timeval);
				if( IsntPGPError( err ) )
					result = sCompareTime( filter, timeval );
			} else for (subkey = key->down; !result && IsntNull(subkey);
							 	subkey = subkey->next)
			{
				if (!pgpKeyDBObjIsReal( subkey ) )
					continue;
				if( pgpObjectType( subkey ) != RINGTYPE_SUBKEY )
					continue;
				err = pgpGetSubKeyTime (subkey,
							(PGPKeyDBObjProperty)filter->value.proptime.prop,
							&timeval);
				if( IsntPGPError( err ) )
					result = sCompareTime( filter, timeval );
			}
			break;
		case kPGPFilterTypeSubKeyBuffer:
			if( filter->filterChildren
				&& IsntNull( subkey = sFindParent( obj, RINGTYPE_SUBKEY ) ) )
			{
				err = pgpGetSubKeyAllocatedPropertyBuffer( subkey,
						(PGPKeyDBObjProperty)filter->value.propbuffer.prop,
						&bufval, &buflen );
				if( IsPGPError( err ) )
					return FALSE;
				result = sCompareString( filter, bufval, buflen );
				PGPFreeData( bufval );
			} else for (subkey = key->down; !result && IsntNull(subkey);
							 	subkey = subkey->next)
			{
				if (!pgpKeyDBObjIsReal( subkey ) )
					continue;
				if( pgpObjectType( subkey ) != RINGTYPE_SUBKEY )
					continue;
				err = pgpGetSubKeyAllocatedPropertyBuffer( subkey,
						(PGPKeyDBObjProperty)filter->value.propbuffer.prop,
						&bufval, &buflen );
				if( IsPGPError( err ) )
					continue;
				result = sCompareString( filter, bufval, buflen );
				PGPFreeData( bufval );
			}
			break;
		case kPGPFilterTypeUserIDBoolean:
			if( filter->filterChildren
				&& IsntNull( userid = sFindParent( obj, RINGTYPE_USERID ) ) )
			{
				err = pgpGetUserIDBoolean (userid,
							(PGPKeyDBObjProperty)filter->value.propbool.prop,
							&boolval);
				if( IsntPGPError( err ) )
					result = sCompareBoolean( filter, boolval );
			} else for (userid = key->down; !result && IsntNull(userid);
							 	userid = userid->next)
			{
				if (!pgpKeyDBObjIsReal( userid) )
					continue;
				if( pgpObjectType( userid ) != RINGTYPE_USERID )
					continue;
				err = pgpGetUserIDBoolean (userid,
							(PGPKeyDBObjProperty)filter->value.propbool.prop,
							&boolval);
				if( IsntPGPError( err ) )
					result = sCompareBoolean( filter, boolval );
			}
			break;
		case kPGPFilterTypeUserIDNumber:
			if( filter->filterChildren
				&& IsntNull( userid = sFindParent( obj, RINGTYPE_USERID ) ) )
			{
				err = pgpGetUserIDNumber (userid,
								(PGPKeyDBObjProperty)filter->value.propnum.prop,
								(PGPInt32 *)&numval);
				if( IsntPGPError( err ) )
					result = sCompareNumber( filter, numval );
			} else for (userid = key->down; !result && IsntNull(userid);
							 	userid = userid->next)
			{
				if (!pgpKeyDBObjIsReal( userid) )
					continue;
				if( pgpObjectType( userid ) != RINGTYPE_USERID )
					continue;
				err = pgpGetUserIDNumber (userid,
								(PGPKeyDBObjProperty)filter->value.propnum.prop,
								(PGPInt32 *)&numval);
				if( IsntPGPError( err ) )
					result = sCompareNumber( filter, numval );
			}
			break;
#if 0
/* Support this when we add a userid time prop function */
		case kPGPFilterTypeUserIDTime:
			if( filter->filterChildren
				&& IsntNull( userid = sFindParent( obj, RINGTYPE_USERID ) ) )
			{
				err = PGPGetUserIDTime (userid, filter->value.proptime.prop,
										   &timeval);
				if( IsntPGPError( err ) )
					result = sCompareTime( filter, timeval );
			} else for (userid = key->down; !result && IsntNull(userid);
							 	userid = userid->next)
			{
				if (!pgpKeyDBObjIsReal( userid) )
					continue;
				if( pgpObjectType( userid ) != RINGTYPE_USERID )
					continue;
				err = PGPGetUserIDTime (userid, filter->value.proptime.prop,
										   &timeval);
				if( IsntPGPError( err ) )
					result = sCompareTime( filter, timeval );
			}
			break;
#endif
		case kPGPFilterTypeUserIDBuffer:
			if( filter->filterChildren
				&& IsntNull( userid = sFindParent( obj, RINGTYPE_USERID ) ) )
			{
				err = pgpGetUserIDAllocatedStringBuffer( userid,
						(PGPKeyDBObjProperty)filter->value.propbuffer.prop,
						&bufval, &buflen );
				if( IsPGPError( err ) )
					break;
				result = sCompareString( filter, bufval, buflen );
				PGPFreeData( bufval );
			} else for (userid = key->down; !result && IsntNull(userid);
							 	userid = userid->next)
			{
				if (!pgpKeyDBObjIsReal( userid) )
					continue;
				if( pgpObjectType( userid ) != RINGTYPE_USERID )
					continue;
				err = pgpGetUserIDAllocatedStringBuffer( userid,
						(PGPKeyDBObjProperty)filter->value.propbuffer.prop,
						&bufval, &buflen );
				if( IsPGPError( err ) )
					continue;
				result = sCompareString( filter, bufval, buflen );
				PGPFreeData( bufval );
			}
			break;
		case kPGPFilterTypeSigBoolean:
			if( filter->filterChildren
				&& IsntNull( sig = sFindParent( obj, RINGTYPE_SIG ) ) )
			{
				err = pgpGetSigBoolean (sig,
							(PGPKeyDBObjProperty)filter->value.propbool.prop,
							&boolval);
				if( IsntPGPError( err ) )
					result = sCompareBoolean( filter, boolval );
			} else for (userid = key->down; !result && IsntNull(userid);
							 	userid = userid->next)
			{
				if (!pgpKeyDBObjIsReal( userid) )
					continue;
			    for (sig = userid->down; !result && IsntNull(sig);
									sig = sig->next)
				{
					if (!pgpKeyDBObjIsReal( sig) )
						continue;
					if( pgpObjectType( sig ) != RINGTYPE_SIG )
						continue;
					err = pgpGetSigBoolean (sig,
								(PGPKeyDBObjProperty)filter->value.propbool.prop,
								&boolval);
					if( IsntPGPError( err ) )
						result = sCompareBoolean( filter, boolval );
				}
			}
			break;
		case kPGPFilterTypeSigNumber:
			if( filter->filterChildren
				&& IsntNull( sig = sFindParent( obj, RINGTYPE_SIG ) ) )
			{
				err = pgpGetSigNumber (sig,
								(PGPKeyDBObjProperty)filter->value.propnum.prop,
								(PGPInt32 *)&numval);
				if( IsntPGPError( err ) )
					result = sCompareNumber( filter, numval );
			} else for (userid = key->down; !result && IsntNull(userid);
							 	userid = userid->next)
			{
				if (!pgpKeyDBObjIsReal( userid) )
					continue;
			    for (sig = userid->down; !result && IsntNull(sig);
									sig = sig->next)
				{
					if (!pgpKeyDBObjIsReal( sig) )
						continue;
					if( pgpObjectType( sig ) != RINGTYPE_SIG )
						continue;
					err = pgpGetSigNumber (sig,
									(PGPKeyDBObjProperty)filter->value.propnum.prop,
									(PGPInt32 *)&numval);
					if( IsntPGPError( err ) )
						result = sCompareNumber( filter, numval );
				}
			}
			break;
		case kPGPFilterTypeSigTime:
			if( filter->filterChildren
				&& IsntNull( sig = sFindParent( obj, RINGTYPE_SIG ) ) )
			{
				err = pgpGetSigTime (sig,
							(PGPKeyDBObjProperty)filter->value.proptime.prop,
							&timeval);
				if( IsntPGPError( err ) )
					result = sCompareTime( filter, timeval );
			} else for (userid = key->down; !result && IsntNull(userid);
							 	userid = userid->next)
			{
				if (!pgpKeyDBObjIsReal( userid) )
					continue;
			    for (sig = userid->down; !result && IsntNull(sig);
									sig = sig->next)
				{
					if (!pgpKeyDBObjIsReal( sig) )
						continue;
					if( pgpObjectType( sig ) != RINGTYPE_SIG )
						continue;
					err = pgpGetSigTime (sig,
								(PGPKeyDBObjProperty)filter->value.proptime.prop,
								&timeval);
					if( IsntPGPError( err ) )
						result = sCompareTime( filter, timeval );
				}
			}
			break;
		case kPGPFilterTypeSigBuffer:
			if( filter->filterChildren
				&& IsntNull( sig = sFindParent( obj, RINGTYPE_SIG ) ) )
			{
				err = pgpGetSigAllocatedPropertyBuffer( sig,
						(PGPKeyDBObjProperty)filter->value.propbuffer.prop,
						&bufval, &buflen );
				if( IsPGPError( err ) )
					break;
				result = sCompareString( filter, bufval, buflen );
				PGPFreeData( bufval );
			} else for (userid = key->down; !result && IsntNull(userid);
							 	userid = userid->next)
			{
				if (!pgpKeyDBObjIsReal( userid) )
					continue;
			    for (sig = userid->down; !result && IsntNull(sig);
									sig = sig->next)
				{
					if (!pgpKeyDBObjIsReal( sig) )
						continue;
					if( pgpObjectType( sig ) != RINGTYPE_SIG )
						continue;
					err = pgpGetSigAllocatedPropertyBuffer( sig,
							(PGPKeyDBObjProperty)filter->value.propbuffer.prop,
							&bufval, &buflen );
					if( IsPGPError( err ) )
						continue;
					result = sCompareString( filter, bufval, buflen );
					PGPFreeData( bufval );
				}
			}
			break;

		case kPGPFilterTypeKeyKeyID:
			pgpKeyID8(key, &pkAlg, &keyID);
			result	= pgpKeyIDsEqual( &keyID, &filter->value.keyKeyID );
			break;

		case kPGPFilterTypeKeyFingerPrint:
		{
			PGPSize		length;
			PGPByte		buffer[32];

			if (pgpKeyV3(key))
			{
				pgpKeyFingerprint16(key, buffer);
				length = 16;
			}
			else
			{
				pgpKeyFingerprint20(key, buffer);
				length = 20;
			}
			if (filter->value.keyFingerPrint.keyFingerPrintLength == length)
			{
				result = !memcmp(buffer,
							filter->value.keyFingerPrint.keyFingerPrintData,
							length);
			}
			break;
		}

		/* Time filters, which utilize shared locals */
		{
			PGPTime		keyTime;
			PGPTime		filterTime;

		case kPGPFilterTypeKeyCreationTime:
			keyTime = pgpKeyCreation(key);
			filterTime = filter->value.keyCreationTime;
			goto compareTimes;

		case kPGPFilterTypeKeyExpirationTime:
			keyTime = pgpKeyExpiration(key);
			filterTime = filter->value.keyExpirationTime;

			/* FALL THROUGH */
		compareTimes:
			if (keyTime == filterTime)
				result = TRUE;	/* All match criteria include equality */
			else if (filter->match == kPGPMatchCriterion_Equal)
				result = FALSE;
			else
			{
				/*
				 * Now we know that the two times are unequal,
				 * and the match criterion is either <= or >=.
				 * So we'll just evaluate <=, and then invert
				 * the result if the criterion was >=.
				 */
				result = (keyTime <= filterTime);
				if (filter->filterType == kPGPFilterTypeKeyExpirationTime)
				{
					if (filterTime == kPGPExpirationTime_Never)
						result = TRUE;
					else if (keyTime == kPGPExpirationTime_Never)
						result = FALSE;
				}

				if (filter->match == kPGPMatchCriterion_GreaterOrEqual)
				{
					result = !result;
				}
				else
				{
					pgpAssert(filter->match == kPGPMatchCriterion_LessOrEqual);
				}
			}
			break;
		}

		case kPGPFilterTypeKeyRevoked:
			result = ((!filter->value.keyRevoked)
					  == (!pgpKeyRevoked(key)));
			break;
		case kPGPFilterTypeKeyDisabled:
			result = ((!filter->value.keyDisabled)
					  == (!pgpKeyDisabled(key)));
			break;

		case kPGPFilterTypeKeyEncryptAlgorithm:
			pgpKeyID8(key, &pkAlg, NULL);
			if( pkAlg == filter->value.keyEncryptAlgorithm )
			{
				result = TRUE;
			} else if( filter->filterChildren
				&& IsntNull( subkey = sFindParent( obj, RINGTYPE_SUBKEY ) ) )
			{
				pgpKeyID8(subkey, &pkAlg, NULL);
				result = (pkAlg == filter->value.keyEncryptAlgorithm);
			} else for (subkey = key->down; !result && IsntNull( subkey );
				 				subkey = subkey->next)
			{
				if (!pgpKeyDBObjIsReal( subkey ) )
					continue;
				if( pgpObjectType( subkey ) != RINGTYPE_SUBKEY )
					continue;
				pgpKeyID8( subkey, &pkAlg, NULL );
				if (pkAlg == filter->value.keyEncryptAlgorithm)
				{
					result = TRUE;
					break;
				}
			}
			break;

		case kPGPFilterTypeKeySigAlgorithm:
			pgpKeyID8(key, &pkAlg, NULL);
			result = (pkAlg == filter->value.keySigAlgorithm);
			break;

		case kPGPFilterTypeKeySubKeyID:
			if( filter->filterChildren
				&& IsntNull( subkey = sFindParent( obj, RINGTYPE_SUBKEY ) ) )
			{
				pgpKeyID8(subkey, NULL, &keyID);
				result = pgpKeyIDsEqual( &keyID, &filter->value.keySubKeyID );
			} else for (subkey = key->down; !result && IsntNull( subkey );
				 				subkey = subkey->next)
			{
				if (!pgpKeyDBObjIsReal( subkey ) )
					continue;
				if( pgpObjectType( subkey ) != RINGTYPE_SUBKEY )
					continue;
				pgpKeyID8(subkey, NULL, &keyID);
				if ( pgpKeyIDsEqual( &keyID, &filter->value.keySubKeyID ) )
				{
					result = TRUE;
					break;
				}
			}
			break;

		case kPGPFilterTypeSigKeyID:
			if( filter->filterChildren
				&& IsntNull( sig = sFindParent( obj, RINGTYPE_SIG ) ) )
			{
				pgpSigID8( sig, NULL, &keyID);
				result = pgpKeyIDsEqual( &keyID, &filter->value.sigKeyID );
			} else for (userid = key->down; !result && IsntNull(userid);
							 	userid = userid->next)
			{
				if (!pgpKeyDBObjIsReal( userid) )
					continue;
			    for (sig = userid->down; !result && IsntNull(sig);
									sig = sig->next)
				{
					if (!pgpKeyDBObjIsReal( sig) )
						continue;
					if( pgpObjectType( sig ) != RINGTYPE_SIG )
						continue;
					pgpSigID8( sig, NULL, &keyID);
					if ( pgpKeyIDsEqual( &keyID, &filter->value.sigKeyID ))
					{
						result = TRUE;
						break;
					}
				}
			}
			break;

		case kPGPFilterTypeKeyEncryptKeySize:
			if( (pgpKeyUse( key ) & PGP_PKUSE_ENCRYPT) )
			{
				comparison = pgpKeyBits(key)
								- filter->value.keyEncryptKeySize;
				result = sComparisonMatchesCriterion(comparison,
													  filter->match);
			} else if( filter->filterChildren
				&& IsntNull( subkey = sFindParent( obj, RINGTYPE_SUBKEY ) ) )
			{
				comparison = pgpKeyBits(subkey)
								- filter->value.keyEncryptKeySize;
				result = sComparisonMatchesCriterion(comparison,
													  filter->match);
			} else for (subkey = key->down; !result && IsntNull( subkey );
				 				subkey = subkey->next)
			{
				if (!pgpKeyDBObjIsReal( subkey ) )
					continue;
				if( pgpObjectType( subkey ) != RINGTYPE_SUBKEY )
					continue;
				comparison = (pgpKeyBits(subkey)
								- filter->value.keyEncryptKeySize);
				if (sComparisonMatchesCriterion(comparison, filter->match))
				{
					result = TRUE;
					break;
				}
			}
			break;

		case kPGPFilterTypeKeySigKeySize:
			comparison = pgpKeyBits(key) - filter->value.keySigKeySize;
			result = sComparisonMatchesCriterion(comparison, filter->match);
			break;

		/* UserID filters, which utilize shared locals */
		{
			char *			string;
			PGPSize			stringLength;

		case kPGPFilterTypeUserIDEmail:
			string = filter->value.userIDEmail;
			stringLength = strlen(string);
			goto stringFilter;

		case kPGPFilterTypeUserIDName:
			string = filter->value.userIDName;
			stringLength = strlen(string);
			goto stringFilter;
			
		case kPGPFilterTypeUserIDString:
			string = filter->value.userIDString; 
			stringLength = strlen(string);

			/* FALL THROUGH */
		stringFilter:
			if( filter->filterChildren
				&& IsntNull( userid = sFindParent( obj, RINGTYPE_USERID ) ) )
			{
				if( pgpUserIDIsAttribute( userid ) )
					break;
				result = sFilterUserID( filter, userid, string, stringLength );
			} else for (userid = key->down; !result && IsntNull(userid);
							 	userid = userid->next)
			{
				if (!pgpKeyDBObjIsReal( userid) )
					continue;
				if( pgpObjectType( userid ) != RINGTYPE_USERID )
					continue;
				if( pgpUserIDIsAttribute( userid ) )
					continue;
				if( sFilterUserID( filter, userid, string, stringLength ) )
				{
					result = TRUE;
					break;
				}
			}
			break;
		}


		/* Compound filters */
		case kPGPFilterTypeNot:
			result = !pgpKeyDBObjMatchesFilter(filter->value.notFilter, obj);
			break;
		case kPGPFilterTypeAnd:
			result = pgpKeyDBObjMatchesFilter(filter->value.andFilter.andFilter1,
											obj)
				  && pgpKeyDBObjMatchesFilter(filter->value.andFilter.andFilter2,
											obj);
			break;
		case kPGPFilterTypeOr:
			result = pgpKeyDBObjMatchesFilter(filter->value.orFilter.orFilter1,
											obj)
				  || pgpKeyDBObjMatchesFilter(filter->value.orFilter.orFilter2,
											obj);
			break;


		default:
			pgpAssertMsg(FALSE, "Unimplemented filter type");
			break;

	}

	/* Cache answer for future sub-object query */
	if( pgpObjectType( obj ) == RINGTYPE_KEY )
	{
		filter->cachedKeyQuery = obj;
		filter->cachedResult = result;
	}

	return result;
}

static PGPError
pgpEncodeSearchTerms( PGPContextRef context,
				char *var,
				PGPSize varlen,
				char **newString )
{
	/* static char const badChars[]   = "*()"; */
	/* We don't want to escape *'s */
	static char const badChars[]   = "()";
	char		*ptrBad;
	char		*ptrGood;
	PGPInt16	i;
	PGPSize		j					= 0;

	*newString = (char*) pgpContextMemAlloc( context,
								2*varlen+1,
								kPGPMemoryMgrFlags_Clear );
	if ( *newString == NULL )
	{
		return kPGPError_OutOfMemory;
	}

	ptrBad  = var;
	ptrGood = *newString;

	while ( ( *ptrBad != '\0' ) && ( j++ < varlen ) )
	{
		for (i=0; badChars[i] != '\0'; i++ )
		{
			if ( *ptrBad == badChars[i] )
			{
				*ptrGood++ = '\\';
				*ptrGood++ = *ptrBad++;
				break;
			}
		}
		if (badChars[i] == '\0')
		{
			*ptrGood++ = *ptrBad++;
		}
	}
	*ptrGood = '\0';

	return kPGPError_NoErr;
}


static PGPError
pgpGrowQueryString(PGPContextRef context,
			char **query, PGPUInt16 *maxsize, PGPUInt16 growthfactor)
{
	PGPError	err;

	if ( strlen(*query) + growthfactor + 1 > *maxsize )
	{
		if (growthfactor < 500)
		{
			growthfactor = 500;
		}
		err = pgpContextMemRealloc( context, (void **)query, 
							*maxsize + growthfactor + 1, 0 );
		if ( IsPGPError(err) )
		{
			return err;
		}
		*maxsize += 500;
	}
	return kPGPError_NoErr;
}

static PGPError
pgpBuildLDAPQuery( 
	PGPFilterRef filter, 
	PGPBoolean *disableVisited,
	char **query, 
	PGPUInt16 *maxsize  )
{
	PGPError	err			= kPGPError_NoErr;
	PGPUInt32	numvalue;
	PGPBoolean	boolvalue;
	PGPTime		timevalue;
	void	   *strvalue;
	PGPSize		strsize;
	char		*newString  = NULL;
	char		keyIDBuffer[ 128 ];
	char		buffer[500];

	switch (filter->filterType)
	{
		case kPGPFilterTypeAnd:
		{
			err = pgpGrowQueryString(filter->context, query, maxsize, 10);
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, "(&" );

			err = pgpBuildLDAPQuery(filter->value.andFilter.andFilter1,
								disableVisited, query, maxsize );
			if (err != kPGPError_NoErr)
			{
				return err;
			}
			err = pgpBuildLDAPQuery(filter->value.andFilter.andFilter2,
								disableVisited, query, maxsize );
			if (err != kPGPError_NoErr)
			{
				return err;
			}
			err = pgpGrowQueryString(filter->context, query, maxsize, 10);
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}

			strcat( *query, ")" );
			break;
		}

		case kPGPFilterTypeOr:
		{
			err = pgpGrowQueryString(filter->context, query, maxsize, 10);
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat(*query, "(|");

			err = pgpBuildLDAPQuery(filter->value.orFilter.orFilter1,
						disableVisited, query, maxsize );
			if (err != kPGPError_NoErr)
			{
				return err;
			}
			err = pgpBuildLDAPQuery(filter->value.orFilter.orFilter2, 
						disableVisited, query, maxsize );
			if (err != kPGPError_NoErr)
			{
				return err;
			}

			err = pgpGrowQueryString(filter->context, query, maxsize, 10);
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, ")");
			break;
		}

		case kPGPFilterTypeNot:
		{
			err = pgpGrowQueryString(filter->context, query, maxsize, 10);
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, "(!" );

			err = pgpBuildLDAPQuery(filter->value.notFilter, 
						disableVisited, query, maxsize );
			if (err != kPGPError_NoErr)
			{
				return err;
			}

			err = pgpGrowQueryString(filter->context, query, maxsize, 10);
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, ")");
			break;
		}

		case kPGPFilterTypeKeyEncryptAlgorithm:
		{	
			numvalue = filter->value.keyEncryptAlgorithm;
buildldap_kPGPFilterTypeKeyEncryptAlgorithm:
			*buffer = '\0';

			switch ( numvalue )
			{
				case kPGPPublicKeyAlgorithm_RSA:
				case kPGPPublicKeyAlgorithm_RSAEncryptOnly:
				case kPGPPublicKeyAlgorithm_RSASignOnly:
				{	
					strcpy( buffer, "(pgpKeyType=RSA)");
					break;
				}
				case kPGPPublicKeyAlgorithm_ElGamal:
				case kPGPPublicKeyAlgorithm_DSA:
				{
					strcpy( buffer, "(pgpKeyType=DSS/DH)");
					break;
				}

				default:
					return kPGPError_InvalidFilterParameter;
			}
			err = pgpGrowQueryString(filter->context, 
								query, maxsize, (PGPUInt16) strlen(buffer));
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, buffer );
			break;

		}
		case kPGPFilterTypeKeySigAlgorithm:
		{	
			numvalue = filter->value.keySigAlgorithm;
buildldap_kPGPFilterTypeKeySigAlgorithm:
			*buffer = '\0';

			switch ( numvalue )
			{
				case kPGPPublicKeyAlgorithm_RSA:
				case kPGPPublicKeyAlgorithm_RSAEncryptOnly:
				case kPGPPublicKeyAlgorithm_RSASignOnly:
				{	
					strcpy( buffer, "(pgpKeyType=RSA)");
					break;
				}
				case kPGPPublicKeyAlgorithm_ElGamal:
				case kPGPPublicKeyAlgorithm_DSA:
				{
					strcpy( buffer, "(pgpKeyType=DSS/DH)");
					break;
				}

				default:
					return kPGPError_InvalidFilterParameter;
			}
			err = pgpGrowQueryString(filter->context, 
								query, maxsize, 
								(PGPUInt16) strlen(buffer));
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, buffer );
			break;
		}
			
		case kPGPFilterTypeKeyKeyID:
		{
			*buffer = '\0';
			err = PGPGetKeyIDString( 
								&filter->value.keyKeyID,
								kPGPKeyIDString_Full,
								keyIDBuffer );
								
			if (IsPGPError(err))
			{
				return err;
			}

buildldap_kPGPFilterTypeKeyKeyID:

			if ( keyIDBuffer[0] == '0' && tolower(keyIDBuffer[1]) == 'x' ) {
				if (strlen(keyIDBuffer) == 10) {
					sprintf( buffer, "(pgpKeyID=%s)", &keyIDBuffer[2]);
				} 
				else 
				{
					sprintf( buffer, "(pgpCertID=%s)", &keyIDBuffer[2]);
				}
			} 
			else 
			{
				if (strlen(keyIDBuffer) == 8) {
					sprintf( buffer, "(pgpKeyID=%s)", keyIDBuffer);
				}
				else
				{
					sprintf( buffer, "(pgpCertID=%s)", keyIDBuffer);
				}
			}

			err = pgpGrowQueryString(filter->context, 
								query, maxsize, (PGPUInt16) strlen(buffer));
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, buffer );
			break;
		}

		case kPGPFilterTypeKeySubKeyID:
		{	
			*buffer = '\0';

			err = PGPGetKeyIDString( 
								&filter->value.keySubKeyID,
								kPGPKeyIDString_Full,
								keyIDBuffer );
								
			if (IsPGPError(err))
			{
				return err;
			}

buildldap_kPGPFilterTypeSubKeyID:

			if ( keyIDBuffer[0] == '0' && tolower(keyIDBuffer[1]) == 'x' ) {
				sprintf( buffer, "(pgpSubKeyID=%s)", &keyIDBuffer[2]);
			} 
			else 
			{
				sprintf( buffer, "(pgpSubKeyID=%s)", keyIDBuffer);
			}

			err = pgpGrowQueryString(filter->context, 
								query, maxsize, (PGPUInt16) strlen(buffer));
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, buffer );
			break;
		}

		case kPGPFilterTypeKeyCreationTime:
		{	
			/*
			 * format: YYYYMMDDHHMMSSZ
			 */

			timevalue = filter->value.keyCreationTime;

buildldap_kPGPFilterTypeKeyCreationTime:

			*buffer = '\0';

			if (timevalue != 0) 
			{
				struct tm	*localTime;
				char		timeString[20];

				localTime = pgpLocalTime(&timevalue);

				if ( localTime == NULL )
				{
					return kPGPError_OutOfMemory;
				}

				sprintf( timeString, "%4d%02d%02d%02d%02d%02dZ",
							1900+localTime->tm_year, localTime->tm_mon+1, 
							localTime->tm_mday, localTime->tm_hour, 
							localTime->tm_min, localTime->tm_sec );

				switch ( filter->match )
				{
					case kPGPMatchCriterion_Equal:

						sprintf( buffer, 
							"(pgpKeyCreateTime=%s)", timeString );
						break;

					case kPGPMatchCriterion_GreaterOrEqual:

						sprintf( buffer, 
							"(|(pgpKeyCreateTime>=%s)(pgpKeyCreateTime=%s))", 
							timeString, timeString);
						break;

					case kPGPMatchCriterion_LessOrEqual:

						sprintf( buffer, 
							"(|(pgpKeyCreateTime<=%s)(pgpKeyCreateTime=%s))", 
							timeString, timeString);
						break;

					default:
						return kPGPError_InvalidFilterParameter;
				}

				err = pgpGrowQueryString(filter->context, 
									query, maxsize, 
									(PGPUInt16) strlen(buffer));
				if ( IsPGPError(err) )
				{
					return kPGPError_OutOfMemory;
				}
			}
			strcat( *query, buffer );
			break;
		}

		case kPGPFilterTypeKeyExpirationTime:
		{	
			/*
			 * format: YYYYMMDDHHMMSSZ
			 */

			timevalue = filter->value.keyExpirationTime;
buildldap_kPGPFilterTypeKeyExpirationTime:

			*buffer = '\0';

			if (timevalue != 0) 
			{
				struct tm	*localTime;
				char		timeString[20];

				localTime = pgpLocalTime(&timevalue);

				if ( localTime == NULL )
				{
					return kPGPError_OutOfMemory;
				}

				sprintf( timeString, "%4d%02d%02d%02d%02d%02dZ",
							1900+localTime->tm_year, localTime->tm_mon+1, 
							localTime->tm_mday, localTime->tm_hour, 
							localTime->tm_min, localTime->tm_sec );

				switch ( filter->match )
				{
					case kPGPMatchCriterion_Equal:

						sprintf( buffer, 
							"(pgpKeyExpireTime=%s)", timeString );
						break;

					case kPGPMatchCriterion_GreaterOrEqual:

						sprintf( buffer, 
							"(|(pgpKeyExpireTime>=%s)(pgpKeyExpireTime=%s))",
							timeString, timeString);
						break;

					case kPGPMatchCriterion_LessOrEqual:

						sprintf( buffer, 
							"(|(pgpKeyExpireTime<=%s)(pgpKeyExpireTime=%s))",
							timeString, timeString );
						break;

					default:
						return kPGPError_InvalidFilterParameter;
				}

				err = pgpGrowQueryString(filter->context, 
									query, maxsize, 
									(PGPUInt16) strlen(buffer));
				if ( IsPGPError(err) )
				{
					return kPGPError_OutOfMemory;
				}
			}
			strcat( *query, buffer );
			break;
		}

		case kPGPFilterTypeKeyRevoked:
		{
			boolvalue = filter->value.keyRevoked;

buildldap_kPGPFilterTypeKeyRevoked:

			sprintf( buffer, "(pgpRevoked=%d)", boolvalue);

			err = pgpGrowQueryString(filter->context, 
								query, maxsize, 
								(PGPUInt16) strlen(buffer));
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, buffer );
			break;
		}

		case kPGPFilterTypeKeyDisabled:
		{
			boolvalue = filter->value.keyDisabled;

buildldap_kPGPFilterTypeKeyDisabled:

			sprintf( buffer, "(pgpDisabled=%d)", boolvalue);

			err = pgpGrowQueryString(filter->context, 
								query, maxsize, 
								(PGPUInt16) strlen(buffer));
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, buffer );
			*disableVisited = TRUE;
			break;
		}

		case kPGPFilterTypeKeyEncryptKeySize:
		{	
			numvalue = filter->value.keyEncryptKeySize;

buildldap_kPGPFilterTypeKeyEncryptKeySize:

			switch ( filter->match )
			{
				case kPGPMatchCriterion_Equal:

					sprintf( buffer, "(pgpKeySize=%05d)", numvalue );
					break;

				case kPGPMatchCriterion_GreaterOrEqual:

					sprintf( buffer, 
							 "(|(pgpKeySize>=%05d)(pgpKeySize=%05d))",
							 numvalue, numvalue );
					break;

				case kPGPMatchCriterion_LessOrEqual:

					sprintf( buffer, 
							 "(|(pgpKeySize<=%05d)(pgpKeySize=%05d))",
							 numvalue, numvalue );
					break;

				default:
					return kPGPError_InvalidFilterParameter;
			}

			err = pgpGrowQueryString(filter->context, 
								query, maxsize, 
								(PGPUInt16) strlen(buffer));
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, buffer );
			break;
		}

		case kPGPFilterTypeUserIDString:
		{	
			strvalue = filter->value.userIDString;
			strsize = strlen( strvalue );

buildldap_kPGPFilterTypeUserIDString:

			err = pgpEncodeSearchTerms( filter->context,
									strvalue, strsize, &newString );

			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}

			if ( filter->match == kPGPMatchCriterion_SubString )
			{
				/* A few special cases before the more general case */
				if( strlen( newString ) == 0 )
					sprintf( buffer, "(pgpUserID=*)" );
				else if( ( strlen( newString ) == 1 ) && ( newString[0] == '*' ) )
					sprintf( buffer, "(pgpUserID=*)" );
				else
					sprintf( buffer, "(pgpUserID=*%s*)", newString );
			} 
			else
			{
				sprintf( buffer, "(pgpUserID=%s)", newString );
			}

			pgpContextMemFree( filter->context, newString);

			err = pgpGrowQueryString(filter->context, 
								query, maxsize, 
								(PGPUInt16) strlen(buffer));
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, buffer );
			break;
		}

		case kPGPFilterTypeUserIDName:
		{	
			strvalue = filter->value.userIDName;
			strsize = strlen( strvalue );

			err = pgpEncodeSearchTerms( filter->context,
									strvalue, strsize, &newString );

			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}

			sprintf( buffer, "(pgpUserID=%s)", newString );

			pgpContextMemFree( filter->context, newString);

			err = pgpGrowQueryString(filter->context, 
								query, maxsize, 
								(PGPUInt16) strlen(buffer));
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, buffer );
			break;
		}

		case kPGPFilterTypeUserIDEmail:
		{	
			strvalue = filter->value.userIDEmail;
			strsize = strlen( strvalue );

buildldap_kPGPFilterTypeUserIDEmail:

			err = pgpEncodeSearchTerms( filter->context,
									strvalue, strsize, &newString);

			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}

			if ( filter->match == kPGPMatchCriterion_SubString )
			{

				sprintf( buffer, "(pgpUserID=*<*%s*>*)", newString );
			} 
			else 
			{
				/*
				 * assume EXACT email address matching
				 */

				sprintf( buffer, "(pgpUserID=*<%s>*)", newString );
			}

			pgpContextMemFree( filter->context, newString);
			err = pgpGrowQueryString(filter->context, 
								query, maxsize, 
								(PGPUInt16) strlen(buffer));
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, buffer );
			break;
		}

		case kPGPFilterTypeSigKeyID:
		{
			/*
			 * This search supports ONLY the LONG 16 char
			 * keyid search
			 */

			*buffer = '\0';

			err = PGPGetKeyIDString( 
							&filter->value.sigKeyID,
							kPGPKeyIDString_Full,
							keyIDBuffer );
							

			if (err != kPGPError_NoErr)
			{
				return err;
			}

buildldap_kPGPFilterTypeSigKeyID:

			if ( keyIDBuffer[0] == '0' && tolower(keyIDBuffer[1]) == 'x' ) {
				sprintf( buffer, "(pgpSignerID=%s)", &keyIDBuffer[2]);
			} 
			else 
			{
				sprintf( buffer, "(pgpSignerID=%s)", keyIDBuffer);
			}

			err = pgpGrowQueryString(filter->context, 
								query, maxsize, 
								(PGPUInt16) strlen(buffer));
			if ( IsPGPError(err) )
			{
				return kPGPError_OutOfMemory;
			}
			strcat( *query, buffer );
			break;
		}

		/*
		 * Generic property searches
		 */
		case kPGPFilterTypeKeyNumber:
			if (filter->value.propnum.prop == kPGPKeyProperty_AlgorithmID)
			{
				numvalue = filter->value.propnum.val;
				goto buildldap_kPGPFilterTypeKeySigAlgorithm;
			}
			else if (filter->value.propnum.prop == kPGPKeyProperty_Bits)
			{
				numvalue = filter->value.propnum.val;
				goto buildldap_kPGPFilterTypeKeyEncryptKeySize;
			}
			break;

		case kPGPFilterTypeKeyTime:
			if (filter->value.proptime.prop == kPGPKeyProperty_Creation)
			{
				timevalue = filter->value.proptime.val;
				goto buildldap_kPGPFilterTypeKeyCreationTime;
			}
			else if (filter->value.proptime.prop == kPGPKeyProperty_Expiration)
			{
				timevalue = filter->value.proptime.val;
				goto buildldap_kPGPFilterTypeKeyExpirationTime;
			}
			break;

		case kPGPFilterTypeKeyBoolean:
			if (filter->value.propbool.prop == kPGPKeyProperty_IsRevoked)
			{
				boolvalue = filter->value.propbool.val;
				goto buildldap_kPGPFilterTypeKeyRevoked;
			}
			else if (filter->value.propbool.prop == kPGPKeyProperty_IsDisabled)
			{
				boolvalue = filter->value.propbool.val;
				goto buildldap_kPGPFilterTypeKeyDisabled;
			}
			break;

		case kPGPFilterTypeKeyBuffer:
			if (filter->value.propbuffer.prop == kPGPKeyProperty_KeyID)
			{
				err = PGPGetKeyIDString( 
							filter->value.propbuffer.val,
							kPGPKeyIDString_Full,
							keyIDBuffer );

				if (err != kPGPError_NoErr)
				{
					return err;
				}
				goto buildldap_kPGPFilterTypeKeyKeyID;
			}
			break;

		case kPGPFilterTypeSubKeyNumber:
			if (filter->value.propnum.prop == kPGPSubKeyProperty_AlgorithmID)
			{
				/* This is correct for keys that have subkeys */
				numvalue = filter->value.propnum.val;
				goto buildldap_kPGPFilterTypeKeyEncryptAlgorithm;
			}
			break;

		case kPGPFilterTypeSubKeyBuffer:
			if (filter->value.propbuffer.prop == kPGPSubKeyProperty_KeyID)
			{
				err = PGPGetKeyIDString( 
							filter->value.propbuffer.val,
							kPGPKeyIDString_Full,
							keyIDBuffer );

				if (err != kPGPError_NoErr)
				{
					return err;
				}
				goto buildldap_kPGPFilterTypeSubKeyID;
			}
			break;

		case kPGPFilterTypeUserIDBuffer:
			if (filter->value.propbuffer.prop == kPGPUserIDProperty_Name)
			{
				strvalue = filter->value.propbuffer.val;
				strsize = filter->value.propbuffer.len;
				goto buildldap_kPGPFilterTypeUserIDString;
			}
			else if (filter->value.propbuffer.prop ==
					 kPGPUserIDProperty_EmailAddress)
			{
				strvalue = filter->value.propbuffer.val;
				strsize = filter->value.propbuffer.len;
				goto buildldap_kPGPFilterTypeUserIDEmail;
			}
			break;

		case kPGPFilterTypeSigBuffer:
			if (filter->value.propbuffer.prop == kPGPSigProperty_KeyID)
			{
				err = PGPGetKeyIDString( 
							filter->value.propbuffer.val,
							kPGPKeyIDString_Full,
							keyIDBuffer );

				if (err != kPGPError_NoErr)
				{
					return err;
				}
				goto buildldap_kPGPFilterTypeSigKeyID;
			}
			break;


		/*
		 * Searches which are not supported by LDAP
		 */

		case kPGPFilterTypeKeyFingerPrint:
		case kPGPFilterTypeKeySigKeySize:
		{
			return kPGPError_UnsupportedLDAPFilter;
		}

		default:
		{
			return kPGPError_UnknownFilterType;
		}
	}

	return(0);

}

static PGPError
pgpBuildLDAPX509Query( 
	PGPFilterRef filter, 
	char **query, 
	PGPUInt16 *maxsize  )
{
	PGPError	err			= kPGPError_NoErr;
	void	   *strvalue;
	PGPSize		strsize;
	char		*newString  = NULL;
	char		buffer[500];

	switch (filter->filterType)
	{
		case kPGPFilterTypeAnd:
		{
			err = pgpGrowQueryString(filter->context, query, maxsize, 10);
			if ( IsPGPError(err) )
				return kPGPError_OutOfMemory;

			strcat( *query, "(&" );

			err = pgpBuildLDAPX509Query(filter->value.andFilter.andFilter1,
								query, maxsize );
			if (err != kPGPError_NoErr)
				return err;

			err = pgpBuildLDAPX509Query(filter->value.andFilter.andFilter2,
								query, maxsize );
			if (err != kPGPError_NoErr)
				return err;

			err = pgpGrowQueryString(filter->context, query, maxsize, 10);
			if ( IsPGPError(err) )
				return kPGPError_OutOfMemory;

			strcat( *query, ")" );
			break;
		}

		case kPGPFilterTypeOr:
		{
			err = pgpGrowQueryString(filter->context, query, maxsize, 10);
			if ( IsPGPError(err) )
				return kPGPError_OutOfMemory;

			strcat(*query, "(|");

			err = pgpBuildLDAPX509Query(filter->value.orFilter.orFilter1,
						query, maxsize );
			if (err != kPGPError_NoErr)
				return err;

			err = pgpBuildLDAPX509Query(filter->value.orFilter.orFilter2, 
						query, maxsize );
			if (err != kPGPError_NoErr)
				return err;

			err = pgpGrowQueryString(filter->context, query, maxsize, 10);
			if ( IsPGPError(err) )
				return kPGPError_OutOfMemory;

			strcat( *query, ")");
			break;
		}

		case kPGPFilterTypeNot:
		{
			err = pgpGrowQueryString(filter->context, query, maxsize, 10);
			if ( IsPGPError(err) )
				return kPGPError_OutOfMemory;

			strcat( *query, "(!" );

			err = pgpBuildLDAPX509Query(filter->value.notFilter, 
						query, maxsize );
			if (err != kPGPError_NoErr)
				return err;

			err = pgpGrowQueryString(filter->context, query, maxsize, 10);
			if ( IsPGPError(err) )
				return kPGPError_OutOfMemory;

			strcat( *query, ")");
			break;
		}

		case kPGPFilterTypeUserIDString:
		{	
			strvalue = filter->value.userIDString;
			strsize = strlen( strvalue );

buildldap_kPGPFilterTypeUserIDString:

			err = pgpEncodeSearchTerms( filter->context,
									strvalue, strsize, &newString );

			if ( IsPGPError(err) )
				return kPGPError_OutOfMemory;

			if ( filter->match == kPGPMatchCriterion_SubString )
			{
				/* A few special cases before the more general case */
				if( strlen( newString ) == 0 )
					sprintf( buffer, "(cn=*)" );
				else if( ( strlen( newString ) == 1 ) && ( newString[0] == '*' ) )
					sprintf( buffer, "(cn=*)" );
				else
					sprintf( buffer, "(cn=*%s*)", newString );
			}
			else
				sprintf( buffer, "(cn=%s)", newString );

			pgpContextMemFree( filter->context, newString);

			err = pgpGrowQueryString(filter->context, 
								query, maxsize, 
								(PGPUInt16) strlen(buffer));
			if ( IsPGPError(err) )
				return kPGPError_OutOfMemory;

			strcat( *query, buffer );
			break;
		}

		case kPGPFilterTypeUserIDName:
		{	
			strvalue = filter->value.userIDName;
			strsize = strlen( strvalue );

			err = pgpEncodeSearchTerms( filter->context,
									strvalue, strsize, &newString );

			if ( IsPGPError(err) )
				return kPGPError_OutOfMemory;

			sprintf( buffer, "(cn=%s)", newString );

			pgpContextMemFree( filter->context, newString);

			err = pgpGrowQueryString(filter->context, 
								query, maxsize, 
								(PGPUInt16) strlen(buffer));
			if ( IsPGPError(err) )
				return kPGPError_OutOfMemory;

			strcat( *query, buffer );
			break;
		}

		case kPGPFilterTypeUserIDEmail:
		{	
			strvalue = filter->value.userIDEmail;
			strsize = strlen( strvalue );

			err = pgpEncodeSearchTerms( filter->context,
									strvalue, strsize, &newString);

			if ( IsPGPError(err) )
				return kPGPError_OutOfMemory;

			if ( filter->match == kPGPMatchCriterion_SubString )
				sprintf( buffer, "(mail=*%s*)", newString );
			else 
			{
				/*
				 * assume EXACT email address matching
				 */

				sprintf( buffer, "(mail=%s)", newString );
			}

			pgpContextMemFree( filter->context, newString);
			err = pgpGrowQueryString(filter->context, 
								query, maxsize, 
								(PGPUInt16) strlen(buffer));
			if ( IsPGPError(err) )
				return kPGPError_OutOfMemory;

			strcat( *query, buffer );
			break;
		}

		/*
		 * Generic property searches
		 */
		case kPGPFilterTypeUserIDBuffer:
			if (filter->value.propbuffer.prop == kPGPUserIDProperty_Name)
			{
				strvalue = filter->value.propbuffer.val;
				strsize = filter->value.propbuffer.len;
				goto buildldap_kPGPFilterTypeUserIDString;
			}
			break;


		/*
		 * Searches which are not supported by LDAP
		 */
		case kPGPFilterTypeSigKeyID:
		case kPGPFilterTypeKeyNumber:
		case kPGPFilterTypeKeyTime:
		case kPGPFilterTypeKeyBoolean:
		case kPGPFilterTypeKeyEncryptAlgorithm:
		case kPGPFilterTypeKeySigAlgorithm:
		case kPGPFilterTypeKeyKeyID:
		case kPGPFilterTypeKeyBuffer:
		case kPGPFilterTypeKeySubKeyID:
		case kPGPFilterTypeKeyCreationTime:
		case kPGPFilterTypeKeyExpirationTime:
		case kPGPFilterTypeKeyRevoked:
		case kPGPFilterTypeKeyDisabled:
		case kPGPFilterTypeKeyEncryptKeySize:
		case kPGPFilterTypeKeyFingerPrint:
		case kPGPFilterTypeKeySigKeySize:
		{
			return kPGPError_UnsupportedLDAPFilter;
		}

		default:
		{
			return kPGPError_UnknownFilterType;
		}
	}

	return(0);

}


PGPError 
PGPLDAPQueryFromFilter( 
	PGPFilterRef	filter,
	char			**queryOut )
{
	PGPUInt16	maxsize	= 1000;
	PGPUInt16	bufLength = 0;
	PGPError	err		= kPGPError_NoErr;
	PGPBoolean	disableVisited = FALSE;
	char		*querybuf;
	
	PGPValidatePtr( queryOut );
	*queryOut	= NULL;
	PGPValidateFilter( filter );

	pgpEnterPGPErrorFunction();

	querybuf = (char*) pgpContextMemAlloc( filter->context, 
								maxsize, kPGPMemoryMgrFlags_Clear );
	if (querybuf == NULL)
	{
		return kPGPError_OutOfMemory;
	}
	*querybuf = '\0';

	err = pgpBuildLDAPQuery(filter, 
						&disableVisited, &querybuf, &maxsize);

	if ( IsntPGPError(err) )
	{
		/*
		 * allocate a buffer big enough to hold the query and potentially
		 * the extra pgpDisabled condition that is added at the end
		 */

		bufLength = strlen(querybuf);
		*queryOut = (char *)PGPNewData(
							PGPPeekContextMemoryMgr( filter->context ),
							bufLength + 41, 0);

		if ( *queryOut != NULL )
		{
			if (disableVisited == FALSE)
			{
				sprintf(*queryOut, "(&%s(pgpDisabled=0))", querybuf );
			}
			else
			{
				pgpCopyMemory( querybuf, *queryOut, bufLength);
				(*queryOut)[bufLength] = '\0';
			}
		}
		else
		{
			*queryOut = NULL;
			err = kPGPError_OutOfMemory;
		}
	}
	else 
	{
		*queryOut = NULL;
	}
	pgpContextMemFree( filter->context, querybuf );

	pgpAssertErrWithPtr( err, *queryOut );
	return err;
}

PGPError 
PGPLDAPX509QueryFromFilter( 
	PGPFilterRef	filter,
	char			**queryOut )
{
	PGPUInt16	maxsize	= 1000;
	PGPUInt16	bufLength = 0;
	PGPError	err		= kPGPError_NoErr;
	char		*querybuf;
	
	PGPValidatePtr( queryOut );
	*queryOut	= NULL;
	PGPValidateFilter( filter );

	pgpEnterPGPErrorFunction();

	querybuf = (char*) pgpContextMemAlloc( filter->context, 
								maxsize, kPGPMemoryMgrFlags_Clear );
	if (querybuf == NULL)
	{
		return kPGPError_OutOfMemory;
	}
	*querybuf = '\0';

	err = pgpBuildLDAPX509Query(filter, 
						&querybuf, &maxsize);

	if ( IsntPGPError(err) )
	{
		bufLength = strlen(querybuf);
		/*
		 * I'd love to tell you what the "+ 41" in the PGPNewData call means
		 * but I really have no idea.  It was there when I copied
		 * PGPLDAPQueryFromFilter, and I don't really want to change it,
		 * for fear that other things start breaking.
		 */
		*queryOut = (char *)PGPNewData(
							PGPPeekContextMemoryMgr( filter->context ),
							bufLength + 41, 0);

		if ( *queryOut != NULL )
		{
			pgpCopyMemory( querybuf, *queryOut, bufLength);
			(*queryOut)[bufLength] = '\0';
		}
		else
		{
			*queryOut = NULL;
			err = kPGPError_OutOfMemory;
		}
	}
	else 
	{
		*queryOut = NULL;
	}
	pgpContextMemFree( filter->context, querybuf );

	pgpAssertErrWithPtr( err, *queryOut );
	return err;
}

static 
PGPError
pgpUrlEncode( PGPContextRef context, char **Dest, char *Source, 
	PGPSize SourceLen )
{
    PGPError	err = kPGPError_NoErr;
    char		*pSource, *pDest;
	PGPSize		len			= 0;

    /*Absolute worst case scenario is three times the source size.  Rather
     *than getting too precise, we'll allocate that much initially, and then
     *realloc it down to actuality later.
     */

    pgpAssert( Source );
    pgpAssert( Dest );

	if( SourceLen == 0 )
		SourceLen = strlen( Source );

    if ( Source && Dest ) 
	{
		*Dest = (char *)pgpContextMemAlloc( context, 
							(SourceLen * 3) + 1,
							kPGPMemoryMgrFlags_Clear );
		if ( Dest != NULL )
		{
			pSource = Source;
			pDest = *Dest;
			
			while(pSource && *pSource && ( len++ < SourceLen ) ) 
			{
				/*Zeroth case:  it's an alphabetic or numeric character*/
				if (!isalpha(*pSource) && 
				    !isdigit(*pSource) && 
				    *pSource != '-') 
				{
					/*First case:  Turn spaces into pluses.*/
					if(*pSource == ' ') 
					{
						*pDest = '+';
					}
					else 
					{
						/*This is overkill, but works for our purposes*/
						*pDest = '%';
						++pDest;
						sprintf(pDest, "%02X", *pSource);
						++pDest;
					}
				}
				else
				{
					*pDest = *pSource;
				}
				
				++pDest;
				++pSource;
			}

			*pDest = '\0';
			
		}
		else
		{
			err = kPGPError_OutOfMemory;
		}
	}
    else
	{
		err = kPGPError_BadParams;
	}
    return err;
}


PGPError 
PGPHKSQueryFromFilter( 
	PGPFilterRef	filter,
	char			**queryOut )
{
	PGPError	err				= kPGPError_NoErr;
	char		*encodedValue	= NULL;
	static const char  prefix[]			= "exact=off&search=";

	PGPValidatePtr( queryOut );
	*queryOut	= NULL;
	PGPValidateFilter( filter );

	pgpEnterPGPErrorFunction();

	/*
	 * HKS searches can only do 1 thing at a time
	 */

	switch ( filter->filterType )
	{
		case kPGPFilterTypeKeyKeyID:
		{
			char	keyIDBuffer[ 128 ];
			
			err = PGPGetKeyIDString( 
							&filter->value.sigKeyID,
							kPGPKeyIDString_Abbreviated,
							keyIDBuffer );
							

			if ( IsPGPError(err) )
			{
				return err;
			}
			*queryOut = (char*) PGPNewData( PGPPeekContextMemoryMgr(
										filter->context ),
										strlen( keyIDBuffer ) +
										strlen( prefix ) +1, 0);
			if ( *queryOut == NULL )
			{
				return kPGPError_OutOfMemory;
			}
			sprintf( *queryOut, "%s%s", prefix, keyIDBuffer);
			break;
		}

		case kPGPFilterTypeKeyBuffer:
		{
			char	keyIDBuffer[ 128 ];
			
			err = PGPGetKeyIDString( 
							filter->value.propbuffer.val,
							kPGPKeyIDString_Abbreviated,
							keyIDBuffer );

			if ( IsPGPError(err) )
			{
				return err;
			}
			*queryOut = (char*) PGPNewData( PGPPeekContextMemoryMgr(
										filter->context ),
										strlen( keyIDBuffer ) +
										strlen( prefix ) +1, 0);
			if ( *queryOut == NULL )
			{
				return kPGPError_OutOfMemory;
			}
			sprintf( *queryOut, "%s%s", prefix, keyIDBuffer);
			break;
		}

		case kPGPFilterTypeUserIDString:
		{
			if ( filter->match == kPGPMatchCriterion_SubString )
			{
				err = pgpUrlEncode( filter->context, &encodedValue, 
							filter->value.userIDString, 
							strlen( filter->value.userIDString ) );

				if ( IsPGPError( err ) )
				{
					return kPGPError_UnsupportedHKPFilter;
				}

				*queryOut = (char*) PGPNewData( PGPPeekContextMemoryMgr(
											filter->context ),
											strlen( encodedValue )+
											strlen( prefix ) +1, 0);
				if ( *queryOut == NULL )
				{
					pgpContextMemFree( filter->context, encodedValue );
					return kPGPError_OutOfMemory;
				}
				sprintf( *queryOut, "%s%s",  prefix, 
					encodedValue);

				pgpContextMemFree( filter->context, encodedValue );
			}
			else
			{
				return kPGPError_UnsupportedHKPFilter;
			}
			break;
		}

		case kPGPFilterTypeUserIDBuffer:
		{
			if ( filter->match == kPGPMatchCriterion_SubString )
			{
				err = pgpUrlEncode( filter->context, &encodedValue, 
							filter->value.propbuffer.val,
							filter->value.propbuffer.len );

				if ( IsPGPError( err ) )
				{
					return kPGPError_UnsupportedHKPFilter;
				}

				*queryOut = (char*) PGPNewData( PGPPeekContextMemoryMgr(
											filter->context ),
											strlen( encodedValue )+
											strlen( prefix ) +1, 0);
				if ( *queryOut == NULL )
				{
					pgpContextMemFree( filter->context, encodedValue );
					return kPGPError_OutOfMemory;
				}
				sprintf( *queryOut, "%s%s",  prefix, 
					encodedValue);

				pgpContextMemFree( filter->context, encodedValue );
			}
			else
			{
				return kPGPError_UnsupportedHKPFilter;
			}
			break;
		}

		case kPGPFilterTypeUserIDEmail:
		{
			if ( filter->match == kPGPMatchCriterion_SubString )
			{
				err = pgpUrlEncode( filter->context, &encodedValue, 
							filter->value.userIDEmail,
							strlen( filter->value.userIDEmail ) );

				if ( IsPGPError( err ) )
				{
					return kPGPError_UnsupportedHKPFilter;
				}

				*queryOut = (char*) PGPNewData( PGPPeekContextMemoryMgr(
											filter->context ),
											strlen( encodedValue )+
											strlen( prefix ) +1, 0);
				if ( *queryOut == NULL )
				{
					pgpContextMemFree( filter->context, encodedValue );
					return kPGPError_OutOfMemory;
				}
				sprintf( *queryOut, "%s%s",  prefix, 
					encodedValue);

				pgpContextMemFree( filter->context, encodedValue );
			}
			else
			{
				return kPGPError_UnsupportedHKPFilter;
			}
			break;
		}

		case kPGPFilterTypeUserIDName:
		{
			if ( filter->match == kPGPMatchCriterion_SubString )
			{
				err = pgpUrlEncode( filter->context, &encodedValue, 
							filter->value.userIDName,
							strlen( filter->value.userIDName ) );

				if ( IsPGPError( err ) )
				{
					return kPGPError_UnsupportedHKPFilter;
				}

				*queryOut = (char*) PGPNewData( PGPPeekContextMemoryMgr(
											filter->context ),
											strlen( encodedValue )+
											strlen( prefix ) +1, 0);
				if ( *queryOut == NULL )
				{
					pgpContextMemFree( filter->context, encodedValue );
					return kPGPError_OutOfMemory;
				}
				sprintf( *queryOut, "%s%s",  prefix, 
					encodedValue);

				pgpContextMemFree( filter->context, encodedValue );
			}
			else
			{
				return kPGPError_UnsupportedHKPFilter;
			}
			break;
		}

		/*
		 * Searches which are not supported by HKP
		 */

		case kPGPFilterTypeNot:
		case kPGPFilterTypeAnd:
		case kPGPFilterTypeOr:
		case kPGPFilterTypeKeySubKeyID:
		case kPGPFilterTypeKeyEncryptAlgorithm:
		case kPGPFilterTypeKeyFingerPrint:
		case kPGPFilterTypeKeyCreationTime:
		case kPGPFilterTypeKeyExpirationTime:
		case kPGPFilterTypeKeyRevoked:
		case kPGPFilterTypeKeyDisabled:
		case kPGPFilterTypeKeySigAlgorithm:
		case kPGPFilterTypeKeyEncryptKeySize:
		case kPGPFilterTypeKeySigKeySize:
		case kPGPFilterTypeSigKeyID:

		case kPGPFilterTypeKeyNumber:
		case kPGPFilterTypeKeyBoolean:
		{
			return kPGPError_UnsupportedHKPFilter;
		}

		default:
		{
			return kPGPError_UnknownFilterType;
			break;
		}
	}
	
	pgpAssertErrWithPtr( err, *queryOut );
	return err;
}

static	const char *	kHexString			=	"0123456789ABCDEF";

static
void
HexEncode(
	const PGPByte *	inBuffer,
	PGPUInt32		inBufSize,
	char *			inOutputBuffer)
{
	char *		p = inOutputBuffer;
	PGPUInt32	i;
	
	for (i = 0; i < inBufSize; i++) {
		*p++ = kHexString[inBuffer[i] >> 4];
		*p++ = kHexString[inBuffer[i] & 0x0F];
	}
	
	*p = '\0';
}

	PGPError
PGPNetToolsCAHTTPQueryFromFilter(
	PGPFilterRef	filter,
	char			**queryOut )
{
	PGPError			err				= kPGPError_NoErr;
	static const char	md5Prefix[]		= "md5=";
	static const char	cnkPrefix[]		= "cnk=";

	PGPValidatePtr( queryOut );
	*queryOut	= NULL;
	PGPValidateFilter( filter );

	pgpEnterPGPErrorFunction();

	switch ( filter->filterType )
	{
		case kPGPFilterTypeKeyFingerPrint:
		{
			*queryOut = (char*) PGPNewData( PGPPeekContextMemoryMgr(
										filter->context ),
										(filter->value.keyFingerPrint.keyFingerPrintLength * 2)
											+ strlen( md5Prefix ) +1, 0);
			if ( *queryOut == NULL )
			{
				return kPGPError_OutOfMemory;
			}
			strcpy( *queryOut, md5Prefix);
			HexEncode(	filter->value.keyFingerPrint.keyFingerPrintData,
						filter->value.keyFingerPrint.keyFingerPrintLength,
						*queryOut + strlen(md5Prefix));
			break;
		}
		
		case kPGPFilterTypeKeyBuffer:
		{
			switch ( filter->value.propbuffer.prop )
			{
				case kPGPKeyProperty_Fingerprint:
				{
					*queryOut = (char*) PGPNewData( PGPPeekContextMemoryMgr(
													filter->context ),
											(filter->value.propbuffer.len * 2)
											+ strlen( md5Prefix ) +1, 0);
					if ( *queryOut == NULL )
					{
						return kPGPError_OutOfMemory;
					}
					strcpy( *queryOut, md5Prefix);
					HexEncode(	filter->value.propbuffer.val,
								filter->value.propbuffer.len,
								*queryOut + strlen(md5Prefix));
					break;
				}
				
				case kPGPKeyProperty_X509MD5Hash:
				{
					*queryOut = (char*) PGPNewData( PGPPeekContextMemoryMgr(
												filter->context ),
												(filter->value.propbuffer.len * 2)
													+ strlen( cnkPrefix ) +1, 0);
					if ( *queryOut == NULL )
					{
						return kPGPError_OutOfMemory;
					}
					strcpy( *queryOut, cnkPrefix);
					HexEncode(	filter->value.propbuffer.val,
								filter->value.propbuffer.len,
								*queryOut + strlen(cnkPrefix));
					break;
				}
				
				default:
				{
					return kPGPError_UnsupportedNetToolsCAFilter;
				}
			}
			break;
		}
			
		default:
		{
			return kPGPError_UnsupportedNetToolsCAFilter;
		}
	}
	
	pgpAssertErrWithPtr( err, *queryOut );
	return err;
}

/*
 * Local Variables:
 * tab-width: 4
 * End:
 * vi: ts=4 sw=4
 * vim: si
 */
